/*
The MIT License (MIT)

Copyright (c) 2015 Sune S. Nielsen, University of Luxembourg

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

package NkBenchSuite;

/**
 * @author ssn
 * This class implements the NKL model allowing removal of ith contribution to the ith calculation.
 */
public class NklModel
{
	/**
	 * @author ssn
	 * This class stores one double values for any array of integers
	 */
	public static class HashTree
	{		
		HashTree[] leafs; // sub leafs of this leaf
		double value; // avlue stored by this leaf		
		int buckets; // the number of values stored for each dimension 
		int dimensions; // dimensions of this tree
		
		static final int maxDimension = 6; // maximum number of dimensions handled per hash level 
		static long instances = 0; // count number of instances for testing purposes
				
		/**
		 * Constructor
		 * @param buckets
		 * @param dimensions
		 */
		public HashTree(int buckets, int dimensions)
		{	
			this.dimensions = Math.min(dimensions, maxDimension);
			this.buckets = buckets;
			if (this.dimensions > 0)
				this.leafs = new HashTree[(int) Math.pow(buckets, this.dimensions)];
			
			this.value = Double.NaN;
			
			NklModel.HashTree.instances++;	
		}		

		/**
		 * Sets a value for the indices
		 * @param indices
		 * @param value
		 */
		public void setValue(int[] indices, double value)
		{
			HashTree leaf = getLeaf(indices, 0);
			leaf.value = value;
		}

		/**
		 * Gets the value previously stored for the indices, or defaultValue if the indices have no value associated 
		 * @param indices
		 * @param defaultValue
		 * @return the value or defaultValue
		 */
		public double getValue(int[] indices, double defaultValue)
		{
			HashTree leaf = getLeaf(indices, 0);
			if (Double.isNaN(leaf.value))
				leaf.value = defaultValue;
			
			return leaf.value;
		}
		
		/**
		 * Gets the index pointed to by the first indices below dimmensions 
		 * @param indices
		 * @param offset
		 * @return
		 */
		private int getIndex(int[] indices, int offset)
		{
			int result = 0;
			
			for (int i=0; i < dimensions; i++)
			{
				int multiplier = (int)Math.pow(this.buckets, i);
				result += indices[i + offset] * multiplier;
			}
			return result;
		}
		
		/**
		 * Gets the leaf selected by the indices. If there are more indices than dimensions, the rest is passed to the child leaf 
		 * @param indices
		 * @param offset
		 * @return the selected leaf
		 */
		private HashTree getLeaf(int[] indices, int offset)
		{
			int index = getIndex(indices, offset);
			int rest = indices.length - offset - this.dimensions;
			if (rest == 0)
			{
				return getLeaf(index, 0);
			}
			else if (rest > 0)
			{
				HashTree leaf = getLeaf(index, rest);				
				return leaf.getLeaf(indices, offset + this.dimensions);
			}
		
			// error
			return null;
		}
		
		/**
		 * Gets or creates the leaf at selected index
		 * @param index
		 * @param leafDimensions
		 * @return the selected leaf
		 */
		private HashTree getLeaf(int index, int leafDimensions)
		{
			if (this.leafs[index] == null)
			{
				this.leafs[index] = new HashTree(this.buckets, leafDimensions);
			}
			return this.leafs[index];
		}
	}
	
	public enum NeighbourhoodType
	{
		UniformRandom,
		TriangularAndUniformRandom,
		Adjacent,
		AdjacentNoCenter,
		AdjacentThreeSpaced,
	}
	
	private int N, K, L;
	
	private boolean[][] adjacencyMatrix;
	private int[][] adjacencyIndex;
	
	private HashTree functionValues;

	private AbstractRandomGenerator random;
	
	/**
	 * Constructor
	 * @param N Length of solutions
	 * @param K Number of neighbors including self (N.B. standard definition counts self by default)
	 * @param L Number of options at each allele or solution position
	 * @param neighborhoodType
	 */
	public NklModel(int N, int K, int L, NeighbourhoodType type)
	{
		this.N = N;
		this.K = K;
		this.L = L;
		
		this.adjacencyMatrix = new boolean[N][N];
		
		this.functionValues = new HashTree(L, K);
		
		this.random = new DefaultRandomGenerator();
		
		this.initNeighborhood(K, type);
		
		NklModel.HashTree.instances = 0;
	}
	
	public void initNeighborhood(int K, NeighbourhoodType type)
	{
		this.K = K;
		this.adjacencyIndex = new int[N][K];
		
		if (type == NeighbourhoodType.AdjacentThreeSpaced)
		{
			for(int i = 0; i < N; i++)
			{	
				this.adjacencyMatrix[i][i] = true;
				
				int pos = i;
				for(int j = 1; j < K; j++)
				{
					pos += (j % 2 == 0 ? -3 * j : 3* j); // select positions right and left spaced by 3 positions
					this.adjacencyMatrix[i][(pos + N) % N] = true;				
				}
			}
		}
		else if (type == NeighbourhoodType.UniformRandom)
		{
			for(int i = 0; i < N; i++)
			{	
				int pos = i;

				for(int j = 0; j < K; j++)
				{					
					pos = this.random.nextInt(N);
					while (this.adjacencyMatrix[i][(pos + N) % N])
						pos = this.random.nextInt(N);
					
					this.adjacencyMatrix[i][(pos + N) % N] = true;
				}
			}
		}
		else if (type == NeighbourhoodType.AdjacentNoCenter)
		{
			//boolean[] repeatMask = {false, true, true};
			//boolean[] repeatMask = {true, true, false};
			boolean[] repeatMask = {true, true, true};
			
			for(int i = 0; i < N; i++)
			{	

				int neighborsSet = 0;
				
				for(int j = 1; neighborsSet < K; j++)
				{
					int pos1 = i + j;
					int pos2 = i - j;
					if (repeatMask[(j - 1) % repeatMask.length])
					{
						this.adjacencyMatrix[i][(pos1 + N) % N] = true;
						this.adjacencyMatrix[i][(pos2 + N) % N] = true;
						neighborsSet+=2;
					}
				}
			}
		}
		else if (type == NeighbourhoodType.TriangularAndUniformRandom)
		{
			for(int i = 0; i < N; i++)
			{	
				int pos = i;
				for(int j = 0; j < K; j++)
				{	
					if (this.random.nextBoolean())
					{
						pos = this.random.nextInt(N);
						while (this.adjacencyMatrix[i][(pos + N) % N])
							pos = this.random.nextInt(N);
						
						this.adjacencyMatrix[i][(pos + N) % N] = true;
					}
					else
					{
						pos = (int)this.random.nextTriangle(i - 10, i, i + 10);
						while (this.adjacencyMatrix[i][(pos + N) % N])
							pos = (int)this.random.nextTriangle(i - 10, i, i + 10);
						
						this.adjacencyMatrix[i][(pos + N) % N] = true;
					}
				}
			}
		}
		else if (type == NeighbourhoodType.Adjacent)
		{
			for(int i = 0; i < N; i++)
			{	
				this.adjacencyMatrix[i][i] = true;
				
				int pos = i;
				for(int j = 1; j < K; j++)
				{
					pos += (j % 2 == 0 ? -j : j); // select positions right and left from i until K-1 have been selected  
					this.adjacencyMatrix[i][(pos + N) % N] = true;				
				}
			}
			
		}
		
		initAdjacencyIndex(K, this.adjacencyMatrix);
	}
	
	public void initAdjacencyIndex(int K, boolean[][] adjacencyMatrix)
	{
		for(int i = 0; i < N; i++)
		{
			int pos = 0;
			for(int j = 0; j < N; j++)
			{
				if (this.adjacencyMatrix[i][j])
				{
					this.adjacencyIndex[i][pos++] = j;					
				}
				System.out.print(this.adjacencyMatrix[i][j] ? "1 " : "0 ");
			}
			System.out.println();
		}
	}
	
	public double evaluate(int[] x)
	{
		double result = 0.0d;
		
		int[] xi = new int[this.K];
		for (int i = 0; i < this.N; i++)
		{
			for (int j = 0; j < this.K; j++)
			{
				xi[j] = x[this.adjacencyIndex[i][j]];	
			}
			
			result += this.functionValues.getValue(xi, this.random.nextDouble());
		}
		return result / N;
	}
	
	public double evaluate(int[] x, int lower, int upper)
	{
		double result = 0.0d;
		int interval = upper - lower;
		
		int[] xi = new int[this.K];
		for (int i = 0; i < this.N; i++)
		{
			for (int j = 0; j < this.K; j++)
			{
				int idx = this.adjacencyIndex[i][j];				
				idx = (idx + interval - lower) % interval + lower;
			
				xi[j] = x[idx];
			}
			
			result += this.functionValues.getValue(xi, this.random.nextDouble());		
		}
		
		return result / N;
	}
	
}
